﻿using System;
using System.ComponentModel;
using System.Linq;
using System.Web.UI;
using System.Web.UI.WebControls;

using Framework.Data;

namespace Framework.Web.UI
{
    /// <summary>
    /// マッピング機能を持ったラジオボタンリストです
    /// </summary>
    [DefaultProperty("Text")]
    [ToolboxData("<{0}:MRadioButtonList runat=server></{0}:MRadioButtonList>")]
    public class MRadioButtonList : RadioButtonList, IMappingControl, IInputControl, ILocalizeControl
    {
        /// <summary>
        /// マッピング名
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string MappingName { get; set; }

        /// <summary>
        /// リストキー
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string ListKey { get; set; }

        [Category("Custom")]
        public event ListSrcRequestEventHandler ListSrcRequest;

        /// <summary>
        /// デフォルト値
        /// </summary>
        [Category("Custom")]
        [DefaultValue("")]
        public string DefaultValue { get; set; }

        /// <summary>
        /// テキスト値をキーにして、テキストをローカライズするかどうか？
        /// </summary>
        [Category("Custom")]
        [DefaultValue("false")]
        public bool LocalizeByText { get; set; }

        /// <summary>
        /// アイテムがローカライズされているときはtrueになります。
        /// </summary>
        private bool _localized = false;

        /// <summary>
        /// OnPreRender
        /// </summary>
        /// <param name="e"></param>
        protected override void OnPreRender(EventArgs e)
        {
            if (base.SelectedIndex < 0 && !string.IsNullOrEmpty(this.DefaultValue))
            {
                if (base.Items.FindByValue(this.DefaultValue) != null)
                {
                    base.SelectedValue = this.DefaultValue;
                }
            }

            base.OnPreRender(e);
        }

        #region IMappingControl メンバ

        /// <summary>
        /// 選択項目をdataに追加します。
        /// </summary>
        /// <param name="data"></param>
        public void RequestMappingData(Framework.Data.MappingData data)
        {
            data.Add(this.MappingName, base.SelectedValue);
        }

        /// <summary>
        /// リスト項目を作成し、その後dataに応じて選択状態を変更します
        /// </summary>
        /// <param name="data"></param>
        public void SetMappingData(Framework.Data.MappingData data)
        {
            base.ClearSelection();

            //リストソースを取得し、データバインド
            if (this.ListKey.IsNotEmpty())
            {
                var listSrc = (ListSrcDataSet)null;
                if (this.ListSrcRequest != null)
                {
                    var e = new ListSrcRequestEventArgs()
                    {
                        ListKey = this.ListKey
                    };
                    this.ListSrcRequest(this, e);
                    listSrc = e.ListSrc;
                }
                else
                {
                    listSrc = ListMgr.GetListSrc(this.ListKey);
                }

                //テキストのローカライズ
                if (this.LocalizeByText == true)
                {
                    listSrc.ListSrc.Foreach(row =>
                    {
                        row.Text = LTextMgr.GetText(row.Text);
                    });

                    _localized = true;
                }

                base.DataValueField = "Value";
                base.DataTextField = "Text";
                base.DataSource = listSrc;
                base.DataBind();
            }

            if (base.Items == null || base.Items.Count == 0)
            {
                return;
            }

            string val = string.IsNullOrEmpty(this.MappingName) ? null : data[this.MappingName];
            if (val == null && this.DefaultValue != null)
            {
                val = this.DefaultValue;
            }

            if (val != null && base.Items.FindByValue(val) != null)
            {
                base.SelectedValue = val;
            }
        }

        #endregion

        #region IInputControl メンバ

        /// <summary>
        /// 必須かどうか
        /// </summary>
        [Category("Custom")]
        public bool IsRequired 
        {
            get { return _isRequired; }
            set { _isRequired = value; } 
        }
        private bool _isRequired = false;

        /// <summary>
        /// 検証結果
        /// </summary>
        public bool IsValid
        {
            get { return _isValid; }
            set { _isValid = value; }
        }
        private bool _isValid = true;

        /// <summary>
        /// 入力不正のコントロールを返します。本コントロールの場合は自身を返します。
        /// </summary>
        /// <returns></returns>
        public Control GetErrorControl()
        {
            return this;
        }

        /// <summary>
        /// 入力された場合にTrueを返します。DefaultValueが指定してある場合は
        /// 選択をDefaultValue以外にした場合にTrueを返します。その他の場合は常にtrueを返します。
        /// </summary>
        /// <returns></returns>
        public bool IsInputed()
        {
            if (base.SelectedIndex < 0)
            {
                return false;
            }
            if (this.DefaultValue.IsNotEmpty())
            {
                return (this.DefaultValue != base.SelectedValue);
            }

            return true;
        }

        /// <summary>
        /// リストが未選択の場合はtrueを返します
        /// </summary>
        /// <returns></returns>
        public bool IsBlank()
        {
            return (base.SelectedIndex < 0);
        }

        /// <summary>
        /// 入力を検証します
        /// </summary>
        /// <param name="dataTypeOnly"></param>
        /// <returns></returns>
        public bool ValidateInput(bool dataTypeOnly)
        {
            if (dataTypeOnly == true) return true;

            //必須チェック
            if (this.IsRequired && this.IsBlank())
            {
                return false;
            }

            return true;
        }

        /// <summary>
        /// 入力不正の場合は、背景色を黄色にします。
        /// </summary>
        /// <param name="isSuccess"></param>
        public void SetValidateResult(bool isSuccess)
        {
            this.IsValid = (this.IsValid && isSuccess);
            base.BackColor = this.IsValid ? UIController.ValidateSuccessColor : UIController.ValidateErrorColor;
        }

        /// <summary>
        /// リストの選択を解除します。DefaultValueが指定してある場合はDefaultValueを選択項目にします。
        /// </summary>
        public void ClearInput()
        {
            if (this.DefaultValue != null)
            {
                if (base.Items.FindByValue(this.DefaultValue) != null)
                {
                    base.SelectedValue = this.DefaultValue;
                }
            }
            else
            {
                base.ClearSelection();
            }
        }

        public string GetInputValue()
        {
            return base.SelectedValue;
        }

        #endregion

        #region ILocalizeControl メンバ

        public void Localize(System.Globalization.CultureInfo lang)
        {
            if (this.LocalizeByText == true && _localized == false)
            {
                //アイテムをテキストの値をキーにローカライズ
                foreach (ListItem item in base.Items)
                {
                    item.Text = LTextMgr.GetText(lang, item.Text);
                }

                _localized = true;
            }
        }

        #endregion
    }
}
